#!/usr/bin/python
# -*- coding: utf-8 -*-
"""Classes and functions for configuration file handling
"""

import sys 
import re
import resource
import directive
import random

RESOURCE_TYPES = ('dird', 'console', 'filed', 'stored')

rxp_item = re.compile('^\\s*(\\w[\\w ]*\\w+)\\s*\\=(.*)') # xxx = xxx matchen
rxp_item2 = re.compile('^\\s*(\\w[\\w ]*\\w+)\\s*\\=\\s*{(.*)}\\s*') # match key = { a = 1; b= 2}
rxp_openbrace = re.compile('^\\s*(\\w[\\w ]*\\w+)\\s*\\=?\\s*\\{\\s*')    # match xxx {
rxp_closebrace = re.compile('[^{]*}.*')               # match }
rxp_comment = re.compile('^\s*#.*')



#'dird', 'console', 'filed' or 'stored'
class Config(object):
    """Class for bacula configuration access"""
    
    def __init__(self, resource_type, filename=""):
        self.resource_type = resource_type
        self.filename = filename
        self.resources = resource.Root(level=0)
        self.resources.name="<root>"

    def read(self):
        self.parse(self.resources,open(self.filename,"r"),0)
        #print self.resources
        
    def write(self, filename=None):
        if(filename==None):
            filename=self.filename
        f=open(filename,"w")
        f.write(str(self.getRoot()))
        f.close()

    def getResourceById(self,theid):
        return self.resources.getById(theid)

    def getResourceByName(self,name):
        return self.resources.getByName(name)

    def getRoot(self):
        return self.resources

    def parse(self,curRes,f,level):
        #print "START",curRes.name

        while True:
            line=f.readline()
            if not line:
                break

            inlinecomment=None
         
            commentStart=line.find("#")
            if commentStart!=-1:
                inlinecomment=line[commentStart:]
                line=line[:commentStart].strip()
                #curRes.add_comment(inlinecomment)

            include=line.strip()
            if include.startswith("@"):
                includepath=include[1:].strip()
                print "include " +includepath
                self.parse(curRes,open(includepath,"r"),level)

            if rxp_closebrace.match(line):
                #print "closebraceline"
                break

            item2 = rxp_item2.match(line)
            if item2:
                #print "item2"
                name = item2.group(1)
                value = item2.group(2)
                #print "item:",name,value
                newRes=resource.Resource(level+1)
                newRes.name=name
                newRes.value="{"+value+"}"
                curRes.add_item(newRes)
                continue

            openbraceline = rxp_openbrace.match(line)
            if openbraceline:
                #print "openbraceline"
                resname = openbraceline.group(1)
                try:
                        resClass = getattr(resource,resname);
                except:
                        resClass = resource.Resource

                newRes=resClass(level+1)
                newRes.name=resname
                curRes.add_item(newRes);
                self.parse(newRes,f,level+1);

                continue

            item = rxp_item.match(line)
            if item:
                name = item.group(1)
                value = item.group(2).strip()
                #print "item:",name,value
                newRes=resource.Resource(level+1)
                newRes.name=name
                newRes.value=value
                curRes.add_item(newRes)
                continue

        #print "END",curRes.name
    
class DirdConfig(Config):
    
    def __init__(self, filename=""):
        Config.__init__(self, 'dird', filename)
        
class ConsoleConfig(Config):
    pass

class FiledConfig(Config):
    def __init__(self, filename=""):
        Config.__init__(self, 'fd', filename)

    def setDefaults(self,directorName,directorPassword,myName):
        director=resource.Director(1,"Director")
        director.add("Name",directorName)
        director.add("Password",'"'+directorPassword+'"')
        self.getRoot().add_item(director)
        
        fileDaemon=resource.FileDaemon(1,"FileDaemon")
        fileDaemon.add("Name",myName)
        #fileDaemon.add("FDport","9102")
        fileDaemon.add("WorkingDirectory","/var/lib/bacula")
        fileDaemon.add("Pid Directory","/var/run")
        fileDaemon.add("Maximum Concurrent Jobs","20")
        self.getRoot().add_item(fileDaemon)

        messages=resource.Messages(1,"Messages")
        messages.add("Name","Standard")
        messages.add("director",directorName+" = all, !skipped, !restored")
        self.getRoot().add_item(messages)
        
class StoredConfig(Config):
    pass

def createClient(clientname,clientaddr,catalog,password):
    newclient=resource.Client(1,"Client")
    newclient.add("Name",clientname)
    newclient.add("Address",clientaddr)
    #newclient.add("FDPort",fdport)
    newclient.add("Catalog",catalog)
    newclient.add("Password",'"'+password+'"')
    return newclient

def genpasswd(len=32):
    charset="01234567890ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz!/%()=?@"
    password=""
    for a in xrange(0,len):
       password += random.choice(charset)
    return password
 
if __name__ == "__main__":

    if len(sys.argv) < 2:
        filename="test.conf"
    else:
        filename=sys.argv[1]   
 
    dirdcfg = DirdConfig(filename)
    dirdcfg.read()

    job=resource.Job(1,"Job")
    job.add("Name",'"test2"')
    job.add("Client",'"test2"')
    job.add("JobDefs",'"testdefs"')
    job.add("FileSet",'"Full Set"')

    root=dirdcfg.getRoot()
    root.add_item(job)

    pwd=genpasswd()
    newclient=createClient("test-fd","testclient","MyCatalog",pwd)
    root.add_item(newclient)

    dirdcfg.write("test.conf.out")

    fdcfg=FiledConfig("test-fd.conf")
    fdcfg.setDefaults("bacula-dird",pwd,"test-fd")
    fdcfg.write()

    sys.exit(0)
