#!/bin/bash

# because the normal mirroring tools often can't access current data
# this scripts updates the local repositories using:
# reposync + createrepo

set -o errexit -o nounset

REPOSYNC_FAILED=""

if [ -r /etc/dass-it/reposync.conf ]; then
   source /etc/dass-it/reposync.conf
fi

usage()
{
  printf "
usage: $0 [directory]

  $0 will scan all subdirectories for repository (*.repo files),
  download the missing packages,
  recreate a repository
  and sign it.

"
}

check_url()
{
   local URL="$1"
   [ -z "$1" ] && return 1
   local CODE=`curl --location --write-out %{http_code} --silent --output /dev/null "$URL"`
   printf "  %s: %s\n" "$URL" "$CODE"
   test "$CODE" = "200"
   return $?
}

STARTDIR=${1:-.}

if ! [ -d $STARTDIR ]; then
  echo "FAILED: $STARTDIR is not a valid directory"
  usage
  exit 1
fi

cd $STARTDIR

ME=reposync-mirror-update
TMPDIR=/tmp/reposync.cache/

REPO_FILES=`find -name "*.repo" ! -name "vermkv-*" | sort`

mkdir -p $TMPDIR
for REPO_FILE in $REPO_FILES; do
    NAME=`sed --quiet -r 's/^\[(.*)\]/\1/p' $REPO_FILE`
    URL=`sed --quiet -r 's/^baseurl=(.*)/\1/ip' $REPO_FILE`
    DIR=`dirname $REPO_FILE`

    echo
    echo
    echo "$ME: $NAME: reposync of $DIR"
    echo

    if ! check_url "$URL"; then
      echo "  SKIPPED: url not accessable"
    else

    # reposync stores in a directory named identical as the name of the repo
    test -L $NAME && rm $NAME
    if test -x $NAME; then
        echo "FAILED: remove $NAME and try again"
        exit 1
    fi

    ln -s $DIR $NAME
    CACHEDIR=`mktemp --directory --tmpdir=$TMPDIR`
    
    # exit on errors
    if ! LOG=$(reposync --config $REPO_FILE --cachedir=$CACHEDIR --delete); then

        printf "FAILED: reposync failed:\n%s" "$LOG" >&2
        REPOSYNC_FAILED="${REPOSYNC_FAILED}${DIR} "

    elif grep "Downloading\|Removing" <<< "$LOG"; then

      test -L $NAME && rm $NAME

      echo
      echo "$ME: $NAME: createrepo"
      createrepo --cachedir .repocache/ $DIR

      echo
      echo "$ME: $NAME: signing"
      rm -f $DIR/repodata/repomd.xml.asc $DIR/repodata/repomd.xml.key
      if [ "$REPOSYNC_GPG_KEY" -a "$REPOSYNC_GPG_PASSPHRASE" ]; then
         gpg --batch --passphrase "$REPOSYNC_GPG_PASSPHRASE" -a --detach-sign --default-key "$REPOSYNC_GPG_KEY" $DIR/repodata/repomd.xml
         gpg --batch --passphrase "$REPOSYNC_GPG_KEY" -a --export "$REPOSYNC_GPG_KEY" > $DIR/repodata/repomd.xml.key
      fi
    fi
    test -L $NAME && rm $NAME
    sleep 1

    fi

done

rm -r $TMPDIR

printf "\n\n"

if [ "$REPOSYNC_FAILED" ]; then
    printf "$ME failed on:\n%s\n" "$REPOSYNC_FAILED"
    exit 1
fi

printf "$ME: done\n"
